unit CompText;

interface

uses Classes, Forms, Graphics, Windows,
     Face;

type
  tText = class(tDrawComponent)
  private
    Points: tPointArray;
  public
    Text, FontName: String;
    FontSize, FontStyle, Rotation: Integer;
    constructor CreateBy(aOwner: tComponent); override;
    constructor CreateByPoint(aOwner: tComponent; aPoint: tPoint); override;
    destructor Destroy; override;
    procedure GetExtent(var aMin, aMax: tPoint); override;
    function GetRgn: hRgn; override;
    procedure Paint; override;
    procedure PutGrips; override;
    procedure SetDefault; override;
    procedure SetPropList; override;
  end;

var dciText: Integer;

implementation

uses Dialogs, SysUtils,
     GDIPAPI, GDIPOBJ;

constructor tText.CreateBy(aOwner: tComponent);
begin
  ImageIndex:= 14;
  inherited CreateByBasicName(aOwner, 'Text');
  Text:= 'This is a text';
  className:= 'Text';
  FontName:= 'Arial';
  FontSize:= 300;
  CreatingByAddPoint:= False;
  Points:= tPointArray.Create(4);
end;

constructor tText.CreateByPoint(aOwner: tComponent; aPoint: Windows.tPoint);
begin
  CreateBy(aOwner);
  Left:= aPoint.X;
  Top:= aPoint.Y;
end;

destructor tText.Destroy;
begin
  Points.Free;
  inherited;
end;

procedure tText.GetExtent(var aMin, aMax: Windows.tPoint);
var i: Integer;
begin
  if Left< aMin.X then aMin.X:= Left;
  if (Left+ Width)> aMax.X then aMax.X:= Left+ Width;
  if Top< aMin.Y then aMin.Y:= Top;
  if (Top+ Height)> aMax.Y then aMax.Y:= Top+ Height;
  for i:= 0 to ComponentCount- 1 do
    tDrawComponent(Components[i]).GetExtent(aMin, aMax);
end;

function tText.GetRgn: hRgn;
begin
  Result:= CreatePolygonRgn(Points.pA^, Points.Count, WINDING);
end;

procedure tText.Paint;
var Matrix: TGPMatrix;
    FontFamily: TGPFontFamily;
    Font: TGPFont;
    SolidBrush: TGPSolidBrush;
    StringFormat: TGPStringFormat;
    myGraphic: TGPGraphics;
    P1: Windows.tPoint;
    NewRect: tGPRectF;
begin
  P1:= DrawFace.Adjust(Point(Left, Top));
  Matrix:= TGPMatrix.Create;
  SolidBrush:= TGPSolidBrush.Create(MakeColor(255, GetBlue(Color), GetGreen(Color), GetRed(Color)));
  FontFamily:= TGPFontFamily.Create(FontName);
  Font := TGPFont.Create(FontFamily, Round(FontSize* DrawFace.ZoomFactor), FontStyleRegular, UnitPixel);
  Matrix.RotateAt(Rotation, MakePoint(P1.x*1.0, P1.y*1.0), MatrixOrderAppend);
  myGraphic:= TGPGraphics.Create(DrawFace.Canvas.Handle);
  myGraphic.SetTransform(matrix);
  myGraphic.DrawString(Text, -1, Font, MakePoint(P1.X*1.0, P1.Y*1.0), SolidBrush);
  StringFormat:= TGPStringFormat.Create;
  StringFormat.SetFormatFlags(StringFormatFlagsNoWrap);
  myGraphic.MeasureString(Text, Length(Text), Font, MakePoint(P1.X* 1.0, P1.Y* 1.0), StringFormat, NewRect);
  Width:= Round(NewRect.Width/DrawFace.ZoomFactor/ 1.03);   //1.03 is a ResultShaper:)
  Height:= Round(NewRect.Height/DrawFace.ZoomFactor/ 1.16); //1.16 is a ResultShaper:)
  Points.SetElem(0, P1);
  Points.SetElem(1, Point(Round(P1.X+ NewRect.Width/1.03), P1.Y));
  Points.SetElem(2, Point(P1.X+ Round(NewRect.Width/1.03), P1.Y+ Round(NewRect.Height/1.16)));
  Points.SetElem(3, Point(P1.X, P1.Y+ Round(NewRect.Height/1.16)));
  myGraphic.TransformPoints(CoordinateSpacePage, CoordinateSpaceWorld, pGPPoint(Points.pA), 4);
  StringFormat.Free;
  Matrix.Free;
  FontFamily.Free;
  Font.Free;
  SolidBrush.Free;
  myGraphic.Free;
  inherited;
end;

procedure tText.PutGrips;
begin
  tTextGrip.CreateTextGrip(DrawFace.GripList, Points, 0);
  tTextGrip.CreateTextGrip(DrawFace.GripList, Points, 1);
  tTextGrip.CreateTextGrip(DrawFace.GripList, Points, 2);
  tTextGrip.CreateTextGrip(DrawFace.GripList, Points, 3);
end;

procedure tText.SetDefault;
begin
  FontSize:= 200;
end;

procedure tText.SetPropList;
begin
  inherited;
  with PropList do
  begin
    Kill('Linecolor');
    Kill('Linewidth');
    Kill('Width');
    Kill('Height');
    Add(tStringProp.CreateProp(@FontName, 'Fontname'));
    Add(tIntegerProp.CreateProp(@FontSize, 'Fontsize'));
    Add(tIntegerProp.CreateProp(@FontStyle, 'Fontstyle(0123)'));
    Add(tIntegerProp.CreateProp(@Rotation, 'Rotation'));
    Default:= tStringProp.CreateProp(@Text, 'Text');
    Add(Default);
  end;
end;

initialization

  //dci -> DrawComponentIndex

  dciText:= Face.DrawCompItemList.Add(tDrawCompItem.Create('Text', tText));

end.
